/*
* Copyright (C) 2004 Ladislav Vitasek
* info@wordrider.net
* http://www.wordrider.net
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
package net.wordrider.dialogs;


import net.wordrider.utilities.Swinger;

import javax.swing.*;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.logging.Logger;

final class ImagePreview extends JComponent implements PropertyChangeListener {
    private ImageIcon thumbnail = null;
    private File file = null;
    private final static Logger logger = Logger.getLogger(ImagePreview.class.getName());

    ImagePreview(final JFileChooser fc) {
        setPreferredSize(new Dimension(100, 50));
        fc.addPropertyChangeListener(this);
    }

    private void loadImage() {
        freeResources();
        if (file == null) {
            return;
        }
        final Image tmpIcon;
        try {
            tmpIcon = Swinger.loadPicture(file);
        } catch (Exception e) {
            logger.warning("Couldn't read the image file for preview :" + file.getPath());
            return;
        }
        //Don't use createImageIcon (which is a wrapper for getResource)
        //because the image we're trying to load is probably not one
        //of this program's own resources.
        //ImageIcon tmpIcon = new ImageIcon(file.getPath());
        if (tmpIcon != null)
            thumbnail = new ImageIcon((tmpIcon.getWidth(this) > 90) ? tmpIcon.getScaledInstance(90, -1, Image.SCALE_FAST) : tmpIcon);
    }

    public final void propertyChange(final PropertyChangeEvent e) {
        boolean update = false;
        final String prop = e.getPropertyName();
        //If the directory changed, don't show an image.
        if (JFileChooser.DIRECTORY_CHANGED_PROPERTY.equals(prop)) {
            file = null;
            update = true;

            //If a file became selected, find out which one.
        } else if (JFileChooser.SELECTED_FILE_CHANGED_PROPERTY.equals(prop)) {
            file = (File) e.getNewValue();
            update = true;
        }

        //Update the preview accordingly.
        if (update) {
            freeResources();
            if (isShowing()) {
                loadImage();
                repaint();
            }
        }
    }

    protected final void paintComponent(final Graphics g) {
        if (thumbnail == null)
            loadImage();

        if (thumbnail != null) {
            int x = getWidth() / 2 - thumbnail.getIconWidth() / 2;
            int y = getHeight() / 2 - thumbnail.getIconHeight() / 2;
            if (y < 0)
                y = 0;
            if (x < 5)
                x = 5;
            thumbnail.paintIcon(this, g, x, y);
        }
    }

    void freeResources() {
        if (thumbnail != null) {
            thumbnail.getImage().flush();
            thumbnail = null;
        }
    }
}
